/*
  Ball Launcher for Bombing Run.
  Author Mark Caldwell aka W@rHe@d of The Reliquary
   
  Used to fire the ball by the player holding the ball.
  Every player is given this weapon when they spawn, but it can't be used until they hold the ball.
  It's better to give every player one instead of spawning one when player gets ball because it's more
  efficient (you don't have the overhead of creating and destroying object)
  and has less issues with switching in and out well.
  
  For this weapon we use a reskinned flak cannon static mesh. We could not use the anim mesh
  because it rejects most of the skins we tried putting on it. A particular skin must be
  designed to work with anim mesh and there is a boolean setting bUsedWithSkeletalMesh which must
  be true but alas is false on a lot of materials. 
*/

class UTBRBallLauncher extends UTWeapon;

var vector DefaultPlayerViewOffset;
var vector DefaultSmallWeaponsOffset;
var DynamicLightEnvironmentComponent DynamicLight;
var DynamicLightEnvironmentComponent RedLight, BlueLight;
var LinearColor DynamicLightColors[2];
var RepNotify UTBRBall TheBall;  //ball currently loaded as ammo
var float AimCone;
var float MaxDist;
var Weapon PendingWeapon;
var Texture2D PassLockImage;
var UTBRPlayerManager PlayerManager;
var float SpecialTossZ;
var bool LetBotShoot;
var int TeamIndex;

simulated function PostBeginPlay()
{
  Super.PostBeginPlay();
  DefaultPlayerViewOffset = PlayerViewOffset;
  DefaultSmallWeaponsOffset = SmallWeaponsOffset;
}

simulated function bool CanThrow()
{
    return false;
}

function SetBringUpTimer()
{
    SetTimer(0.1, true, 'CheckIfActive');
}

//used to counter a problem where somtimes you can switch out to another weapon while ball launcher is first switching in
function CheckIfActive()
{
    if (
        (Instigator != none) && (Instigator.Weapon != self) && (Instigator.InvManager.PendingWeapon != self) && 
        (! bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons))
        )
    {
        Instigator.SetActiveWeapon(self);
    }       
       
    if ((Instigator.Weapon == self) || (TheBall == none) || bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons))
    {
        ClearTimer('CheckIfActive');
    }
}

function bool DenyPickupQuery(class<Inventory> ItemClass, Actor Pickup)
{
	//because deployables force switchin must deny pickup if can't switchout launcher
	return HasAnyAmmo() && ClassIsChildOf(ItemClass, class'UTDeployable') && 
           (! bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons));
}

/** called on both Instigator's current weapon and its pending weapon (if they exist)
 * @return whether Instigator is allowed to switch to NewWeapon
 */
simulated function bool AllowSwitchTo(Weapon NewWeapon)
{
    //UTDeployable weapons don't allow to be switched out until they are deployed 
	return bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons) || (UTDeployable(NewWeapon) == none);
}

simulated function Activate()
{   
    super.Activate();

    ClearTimer('InactiveTimer');
  
    if (bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons))
    {
        InventoryGroup=1;
        Priority=100;
    }
}

function ConsumeAmmo( byte FireModeNum )
{
}

//shoot the ball
simulated function Projectile ProjectileFire()
{
    local vector RealStartLoc, direction;

    if( Role == ROLE_Authority )
    {       
        if (TheBall == none)
        {
            return none;
        }
        
        if (TheBall.HolderPRI.bBot)
        {
            if (! LetBotShoot)
            {
                //bots are trigger happy and they will pass ball back to enemy accidentally unless
                //we only let them shoot when triggered by certain ai functions
                return none;
            }
            
            //stops bot from firing next weapon at teammate after passing ball
            StopFire(0);
        }

        LetBotShoot = false;        
        RealStartLoc = GetPhysicalFireStartLoc();
        direction = Vector(GetAdjustedAim( RealStartLoc ));
        PlayerManager.TheBall = TheBall;
        TheBall.ShotByBallLauncher = true;
        TheBall.Throw(RealStartLoc, direction);
                                
        OnBallDetached();      
    }
    
    return none;
} 

function OnBallDetached()
{
    TheBall = none;
    InventoryGroup=0;
    Priority=1;    
          
    if ((PendingWeapon != none) && (Instigator != none))
    {
        Instigator.SetActiveWeapon(PendingWeapon);
    }
    else
    {    
        PutDownWeapon();
    }
}

//used for bots (UTBRSquadAI.CheckBotShooting) to get pass lock rather than calling StartFire(1).
//because StartFire uses states, you can't just call StartFire(1) and then StartFire(0),
//because it would kick gun into state for StartFire(1) and then StartFire(0) call is ignored.
//calling GetPassLock and then StartFire(0) works.
function GetPassLock()
{
    CustomFire();
}

//used to get a pass lock on a team mate using alt fire. when fired, ball will seek out that player.
simulated function CustomFire()
{
    local Pawn p;
    local Pawn bestP;
    local float bestWt;
    local vector toPawn;
    local vector eyeDir;
    local float cosdot;
    local float curwt;
    local float curdist;
    local float curdot;
    local UTPlayerReplicationInfo pri;


    if ( (Instigator.Role < ROLE_Authority) || (Instigator.PlayerReplicationInfo == None) || (Instigator.PlayerReplicationInfo.Team == None) )
        return;

    if (TheBall == none)
    {
        return;
    }     
      
    bestP = None;
    cosdot = cos(AimCone*PI/180);
    eyeDir = Vector(Instigator.Controller.Rotation);
    bestWt = 0.0;

    foreach Instigator.VisibleActors(class'Pawn', p, MaxDist )
    {
        if (UTVehicle(p) != none)
        {
            p = UTVehicle(p).Driver;
        }
        
        pri = UTBRGame(WorldInfo.Game).GetPlayerReplicationInfo(p);
            
        if ( (p == Instigator) || (pri == None) || (pri.Team != Instigator.PlayerReplicationInfo.Team) )
            continue;

               
        toPawn = p.Location - Instigator.Location;
        curdot = Normal(toPawn) dot eyeDir;
        curdist = VSize(toPawn);

        if( curdot < cosdot )
            continue;
        if( curdist > MaxDist )
            continue;       

        curwt = 1.0 - (curDist / MaxDist);
        curwt *= curdot;
        if( curwt > bestWt )
        {
            bestWt = curwt;
            bestP = p;
        }
    }

    SetPassTarget(bestP);
    StopFire(1);
}

function Controller GetController(Pawn P)
{   
    return UTBRGame(WorldInfo.Game).GetController(P);
}

//called to set the target of a pass lock
function SetPassTarget( Pawn passTarg )
{
    local Controller ptc;
    
    ptc = GetController(TheBall.PassTarget);
    
    if ( (TheBall.PassTarget != None) && (TheBall.PassTarget != PassTarg) && (PlayerController(ptc) != None) )
        PlayerController(ptc).ClientPlaySound(TheBall.LockLostSound);
         
    TheBall.PassTarget = passTarg;
    ptc = GetController(TheBall.PassTarget);    
    
    if ( PlayerController(Instigator.Controller) != None )
    {
        if ( passTarg != None )
            PlayerController(Instigator.Controller).ClientPlaySound(TheBall.LockAcquiredSound);
        else
            PlayerController(Instigator.Controller).ClientPlaySound(TheBall.LockLostSound);
    }
    if ( (TheBall.PassTarget != None) && (PlayerController(ptc) != None) )
        PlayerController(ptc).ClientPlaySound(TheBall.LockAcquiredSound); 
    
}

simulated function bool HasAnyAmmo()
{
    return TheBall != none;
}

simulated function PutDownWeapon()
{
    if ((!HasAnyAmmo()) || bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons))
    {
        GotoState('WeaponPuttingDown');
    }
    else
    {
        PendingWeapon = Instigator.InvManager.PendingWeapon;
    }
}

auto simulated state Inactive
{
    function InactiveTimer()
    {
         //when holder fires ball, this will make it will switch back to the weapon they were holding
         //before bring up the ball launcher.
         //only an issue when AllowBallCarrierWeapons is true, because holder can switch weapons freely
         //and we need to know what weapon they were holding before switching to launcher.
         PendingWeapon = Instigator.Weapon;
    }
    
    simulated function BeginState( Name PreviousStateName )
    {
        super.BeginState(PreviousStateName);
             
        if (bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons))
        {
            SetTimer(0.25, true, 'InactiveTimer');
        }
    }

    simulated function EndState( Name NextStateName )
    {
        super.EndState(NextStateName);
        ClearTimer('InactiveTimer');        
    }       
}

/**
 * Access to HUD and Canvas.
 * Event always called when the InventoryManager considers this Inventory Item currently "Active"
 * (for example active weapon)
 *
 * @param   HUD         - HUD with canvas to draw on
 */
simulated function ActiveRenderOverlays( HUD H )
{
    Super.ActiveRenderOverlays(H);
    
    DrawTargeting(H.Canvas);   
    
    if ((UTBRGame(WorldInfo.Game) != none) && UTBRGame(WorldInfo.Game).DebugHud)
    {
        UTBRGame(WorldInfo.Game).SendDebugMessage("BallLauncher ActiveRenderOverlays: Hud=" $ H $ ",Instigator=" $ Instigator $ ",Controller=" $ Instigator.Controller $ ", bNoCrosshair=" $ UTPlayerController(Instigator.Controller).bNoCrosshair);    
    }
}

//draw pass lock graphics upon the pass locked team mate
simulated function DrawTargeting( Canvas C )
{
    local int XPos, YPos;
    local vector ScreenPos;
    local vector X,Y,Z,Dir;
    local float RatioX, RatioY;
    local float tileX, tileY;
    local float Dist;

    local float SizeX;
    local float SizeY;
    local float TargetingSize;

    if ((TheBall == none) || (TheBall.PassTarget == none))
    {
        return;
    }
      
    TargetingSize = 1;
    
    SizeX = TargetingSize * 96.0;
    SizeY = TargetingSize * 96.0;

    ScreenPos = C.Project( TheBall.PassTarget.Location );

    RatioX = C.SizeX / 640.0;
    RatioY = C.SizeY / 480.0;

    tileX = sizeX * RatioX;
    tileY = sizeY * RatioY;

    GetAxes(Instigator.Rotation, X,Y,Z);
    Dir = TheBall.PassTarget.Location - Instigator.Location;
    Dist = VSize(Dir);
    Dir = Dir/Dist;

    if ( (Dir Dot X) > 0.6 ) // don't draw if it's behind the eye
    {
        XPos = ScreenPos.X;
        YPos = ScreenPos.Y;
        C.DrawColor.R = 255;
        C.DrawColor.G = 255;
        C.DrawColor.B = 255;
        C.DrawColor.A = 255;
        C.SetPos(XPos - tileX*0.5, YPos - tileY*0.5);

        //C.DrawTile(CrosshairImage,tileX, tileY, LockedCrossHairCoordinates.U, LockedCrossHairCoordinates.V, LockedCrossHairCoordinates.UL,LockedCrossHairCoordinates.VL);
        C.DrawTile(PassLockImage,tileX, tileY, LockedCrossHairCoordinates.U, LockedCrossHairCoordinates.V, LockedCrossHairCoordinates.UL,LockedCrossHairCoordinates.VL);
    }
}

//adjust weapon position based on player fov.
//must be done because we do not use anim mesh.  
simulated function AdjustForFov()
{
    local vector v;
    
    /*
       unfortunately static meshes do not have a SetFov() function like anim meshes do
       so we must manually adjust offset according to the fov.
       
       the trick is to find good settings which work at all fov's with only an adjustment
       to X. Tough to find good settings because there is a glitch in lower right corner
       of screen which wipes out the back of the gun, so tricky to position gun in place
       where glitch is avoided or minimized.
        
       good PlayerViewOffset settings which we use as the basis for below calculations:       
       fov  80   23.0, 12.0, -10.0     normal 3d drawscale
       fov 100   16.5, 12.0, -10.0     x=0.75 3d drawscale
       
       good SmallWeaponsOffset settings:
       fov 80    5.0, 4.0, -3.0    normal 3d drawscale
       fov 100   3.5, 4.0, -3.0    x=0.75 3d drawscale
    */
    
    if ((Instigator != none) && (PlayerController(Instigator.Controller) != none) && (DefaultPlayerViewOffset.X != 0))
    {
        //gun stretches as fov is increased, so reduce length of gun
        //X=0.75 is a good setting at fov 100, X=1 at fov 80.
        v.x = 1;
        v.y = 1;
        v.z = 1;
        v.x =
        1.0 +
        ((80 - PlayerController(Instigator.Controller).FovAngle) * ((1.0 - 0.75) / 20.0));              
        SetDrawScale3D(v);
           
        PlayerViewOffset.X = 
        DefaultPlayerViewOffset.X + 
        ((80 - PlayerController(Instigator.Controller).FovAngle) * ((23.0 - 16.5) / 20.0));
        
        SmallWeaponsOffset.X =         
        DefaultSmallWeaponsOffset.X + 
        ((80 - PlayerController(Instigator.Controller).FovAngle) * ((5.0 - 3.5) / 20.0));                
    }
}

simulated event SetPosition(UTPawn Holder)
{
   AdjustForFov();
   super.SetPosition(Holder);
}

simulated function SetSkin(Material NewMaterial)
{       
    if (bool(UTBRGame(WorldInfo.Game).Settings.NecrisTheme))
    {
        Mesh.SetMaterial(0, Material'UN_Liquid.SM.Materials.M_UN_Liquid_SM_NanoBlack_03_Master');  
    }
    else
    {
        Mesh.SetMaterial(0, Material'LT_Deco.SM.Materials.M_LT_Deco_SM_Holodisplay01_B');          
    }
    
    //UTPawn(Instigator).GetTeamNum() and Instigator.Controller.GetTeamNum() do not work. returns 255 or Controller is None just after getting off hoverboard (not the case for other vehicles)
    
    if (TeamIndex == 0)
    {
        Mesh.SetLightEnvironment(RedLight);
    }
    else
    {
        Mesh.SetLightEnvironment(BlueLight);
    }    
}

simulated event ReplicatedEvent(name VarName)
{
    if ((VarName == 'TheBall') && (TheBall == none))
    {
        PutDownWeapon();
    }
    
    super.ReplicatedEvent(VarName);
}

//testing only
simulated exec function BLToggleSW()
{
    bSmallWeapons = ! bSmallWeapons;
}

//testing only
simulated exec function BLSetFov(float fov)
{
     PlayerController(Instigator.Controller).SetFov(fov);
}

//testing only
simulated exec function BLSetDS(float ds)
{
  SetDrawScale(ds);
}

//testing only.
//turn off adjusts for fov made by AdjustForFov()
simulated exec function BLNoAdjustFov()
{
   DefaultPlayerViewOffset.X = 0; 
}

//testing only
simulated exec function BLSetSW(float POSX, float POSY, float POSZ)
{
   local Vector v;
   
   v = vect(0,0,0);
   v.x = POSX;
   v.y = POSY;
   v.z = POSZ;
   SmallWeaponsOffset = v;
   DefaultSmallWeaponsOffset = v;   
}

//testing only
simulated exec function BLSetPV(float POSX, float POSY, float POSZ)
{
   local Vector v;
   
   v = vect(0,0,0);
   v.x = POSX;
   v.y = POSY;
   v.z = POSZ;
   PlayerViewOffset = v;
   DefaultPlayerViewOffset = v;   
}

//testing only
simulated exec function BLSetDS3D(float X, float Y, float Z)
{
   local Vector v;
   
   v = vect(0,0,0);
   v.x = X;
   v.y = Y;
   v.z = Z;
   SetDrawScale3D(v);
}


replication
{
    if (Role==ROLE_Authority)
        TheBall, TeamIndex;
}

defaultproperties
{
    DrawScale=0.7
    EquipTime=0.0
    PutDownTime=0.0 
    IconX=458
    IconY=83
    IconWidth=31
    IconHeight=45   
    IconCoordinates=(U=453.000000,V=327.000000,UL=135.000000,VL=57.000000)
    CrossHairCoordinates=(U=64.000000,V=0.000000)   
    Priority=1
    ItemName="Ball Launcher"
    PickupMessage="Ball Launcher"
    InventoryGroup=0    
    //Mesh=none 
    AimCone=20.0
    MaxDist=30000.0    
    AmmoCount=1
    MaxAmmoCount=1  
    WeaponFireSnd(0)=none   
    WeaponFireSnd(1)=none   
    FireOffset=(X=30.000000,Y=0.000000,Z=0.000000)   
    GroupWeight=0.71   
    Name="Default__UTBRBallLauncher"
    ObjectArchetype=UTWeapon'Default__UTWeapon'
    WeaponFireTypes(0)=EWFT_Projectile
    WeaponFireTypes(1)=EWFT_Custom
    FireInterval(0)=0.01
    FireInterval(1)=0.01   
    //PassLockImage=Texture2D'BombingRunGraphics.PassTargetFocus'
    //LockedCrossHairCoordinates=(U=0.000000,V=0.000000,UL=300.000000,VL=300.000000)   
    PassLockImage=Texture2D'UI_HUD.HUD.UTCrossHairs'
    LockedCrossHairCoordinates=(U=406.000000,V=320.000000,UL=76.000000,VL=77.000000)    
    
    
    //this needed to stop mutators like instagib removing launcher from game. See actor PreBeginPlay()
    bGameRelevant=true
    
    
    Begin Object Class=DynamicLightEnvironmentComponent Name=BLLightEnvironment ObjName=BLLightEnvironment Archetype=DynamicLightEnvironmentComponent'Engine.Default__DynamicLightEnvironmentComponent'
      bDynamic=True
      bCastShadows=False        
      AmbientGlow=(R=1.000000,G=1.000000,B=1.000000,A=1.000000)
      Name="BLLightEnvironment"
      ObjectArchetype=DynamicLightEnvironmentComponent'Engine.Default__DynamicLightEnvironmentComponent'
    End Object
    DynamicLight=BLLightEnvironment
    
    DynamicLightColors(0)=(B=1,G=1,R=2,A=1)
    DynamicLightColors(1)=(B=2,G=1,R=1,A=1)

    Begin Object Class=StaticMeshComponent Name=BLMesh ObjName=BLMesh Archetype=StaticMeshComponent'Engine.Default__StaticMeshComponent'   
      StaticMesh=StaticMesh'WP_FlakCannon.Mesh.S_WP_Flak_3P'
      DepthPriorityGroup=SDPG_Foreground
      Rotation=(Yaw=-16384)
      bUseAsOccluder=False
      CastShadow=False
      Name="BLMesh"
      ObjectArchetype=StaticMeshComponent'Engine.Default__StaticMeshComponent'
      bOnlyOwnerSee=True
    End Object    
              
    Begin Object Name=FirstPersonMesh ObjName=FirstPersonMesh Archetype=UTSkeletalMeshComponent'UTGame.Default__UTWeapon:FirstPersonMesh'
      FOV=70.000000
      SkeletalMesh=SkeletalMesh'WP_FlakCannon.Mesh.SK_WP_FlakCannon_1P'   
      AnimTreeTemplate=AnimTree'WP_FlakCannon.Anims.AT_FlakCannon'
      AnimSets(0)=AnimSet'WP_FlakCannon.Anims.K_WP_FlakCannon_1P_Base'
      ObjectArchetype=UTSkeletalMeshComponent'UTGame.Default__UTWeapon:FirstPersonMesh'
    End Object
    
    //Mesh=FirstPersonMesh 
    Mesh=BLMesh    
    
    Begin Object Name=PickupMesh ObjName=PickupMesh Archetype=SkeletalMeshComponent'UTGame.Default__UTWeapon:PickupMesh'
      SkeletalMesh=SkeletalMesh'WP_FlakCannon.Mesh.SK_WP_FlakCannon_3P_Mid'
      ObjectArchetype=SkeletalMeshComponent'UTGame.Default__UTWeapon:PickupMesh'
    End Object 
    DroppedPickupMesh=PickupMesh
    PickupFactoryMesh=PickupMesh   
    AttachmentClass=Class'BombingRun.UTBRBallLauncherAttachment'   
    PlayerViewOffset=(X=23.0,Y=12.0,Z=-10.0)
    SmallWeaponsOffset=(X=5.0,Y=4.0,Z=-3.0)    
    
   Begin Object Class=DynamicLightEnvironmentComponent Name=Light0 ObjName=Light0 Archetype=DynamicLightEnvironmentComponent'Engine.Default__DynamicLightEnvironmentComponent'
      bDynamic=True
      bCastShadows=False        
      AmbientGlow=(R=5.000000,G=0.000000,B=0.000000,A=1.000000)
      Name="Light0"
      ObjectArchetype=DynamicLightEnvironmentComponent'Engine.Default__DynamicLightEnvironmentComponent'
   End Object
   RedLight=Light0
   
   Begin Object Class=DynamicLightEnvironmentComponent Name=Light1 ObjName=Light1 Archetype=DynamicLightEnvironmentComponent'Engine.Default__DynamicLightEnvironmentComponent'
      bDynamic=True
      bCastShadows=False        
      AmbientGlow=(R=0.000000,G=0.000000,B=5.000000,A=1.000000)
      Name="Light1"
      ObjectArchetype=DynamicLightEnvironmentComponent'Engine.Default__DynamicLightEnvironmentComponent'
   End Object      
   BlueLight=Light1     
   
   Components(0)=Light0
   Components(1)=Light1

   //static meshes do not support animations so we don't use them.
   //we can't use the hand meshes either because their position is
   //based on attaching to bone of anim mesh, and you can't do this on
   //a static mesh. hand mesh positioning ends up then being whatever
   //they were on last weapon held before activating launcher.
   
   //ArmsAnimSet=AnimSet'WP_FlakCannon.Anims.K_WP_FlakCannon_1P_Arms'
   ArmsAnimSet=none     
   WeaponFireAnim(0)=""
   WeaponFireAnim(1)=""
   ArmFireAnim(0)=""
   ArmFireAnim(1)=""
   WeaponPutDownAnim=""
   ArmsPutDownAnim=""
   WeaponEquipAnim=""
   ArmsEquipAnim=""
   WeaponIdleAnims(0)=""
   ArmIdleAnims(0)=""
}
